#!/usr/bin/env python3
import argparse
import csv
import json
import os
from statistics import mean

import cv2
import numpy as np
from PIL import Image

def clamp(x, lo=0, hi=100):
    return max(lo, min(hi, x))

def compute_time_factor(opened_days: float, stored_days: float | None):
    """
    """
    score = 100.0 - opened_days * 10.0
    reasons = []
    reasons.append(f"Opened for {opened_days:.1f} day(s)")
    if stored_days is not None:
        score -= stored_days * 2.0
        reasons.append(f"Stored for {stored_days:.1f} day(s)")
    return clamp(score), reasons

def compute_env_factor(csv_path: str):
    """
    """
    temps = []
    hums = []
    gases = []
    door_opens = 0
    n = 0

    LAST_N = 120

    with open(csv_path, "r", encoding="utf-8", errors="ignore") as f:
        rows = list(csv.DictReader(f))
        if not rows:
            return 70, ["No env data found (default env score)"]

        tail = rows[-LAST_N:] if len(rows) > LAST_N else rows

        for r in tail:
            n += 1

            def get_float(*keys, default=None):
                for k in keys:
                    if k in r and r[k] not in (None, ""):
                        try:
                            return float(r[k])
                        except:
                            pass
                return default

            def get_str(*keys, default=""):
                for k in keys:
                    if k in r and r[k] is not None:
                        return str(r[k])
                return default

            t = get_float("temp", "temperature", "t", default=None)
            h = get_float("humidity", "hum", "h", default=None)
            g = get_float("gas_raw", "gas", "mq2", default=None)

            door = get_str("door_open", "door", default="false").lower()
            door_is_open = door in ("true", "1", "yes", "open")

            if t is not None: temps.append(t)
            if h is not None: hums.append(h)
            if g is not None: gases.append(g)
            if door_is_open: door_opens += 1

    t_avg = mean(temps) if temps else 5.0
    h_avg = mean(hums) if hums else 70.0
    g_avg = mean(gases) if gases else 800.0

    score = 100.0
    reasons = []

    if t_avg > 8:
        score -= (t_avg - 8) * 6
        reasons.append(f"Avg temp {t_avg:.1f}°C is high")

    if h_avg > 80:
        score -= (h_avg - 80) * 1.2
        reasons.append(f"Avg humidity {h_avg:.0f}% is high")

    if g_avg > 2000:
        score -= 30
        reasons.append(f"Gas level is poor (avg {g_avg:.0f})")
    elif g_avg > 900:
        score -= 12
        reasons.append(f"Gas level is moderate (avg {g_avg:.0f})")

    ratio = (door_opens / n) if n else 0
    if ratio > 0.2:
        score -= 10
        reasons.append(f"Frequent door openings ({ratio*100:.0f}% of samples)")

    if not reasons:
        reasons.append("Environment looks stable")

    return clamp(score), reasons

def compute_visual_factor(img_path: str | None):
    """
    Compute visual freshness cues from an uploaded milk image.
    Uses interpretable features: brightness, color shift, and sharpness.
    Returns:
        score (0–100),
        explanation list (for XAI layer)
    """

    # ---------- 缺省情况 ----------
    if not img_path:
        return 70, ["No photo uploaded (visual factor default)"]

    if not os.path.exists(img_path):
        return 60, ["Photo path missing (visual factor lowered)"]

    explanations = []
    score = 100  # 从满分开始扣分（更直观、好解释）

    # ---------- 读取图像 ----------
    img = Image.open(img_path).convert("RGB")
    img_np = np.array(img)

    # ---------- 1. 亮度 ----------
    gray = cv2.cvtColor(img_np, cv2.COLOR_RGB2GRAY)
    brightness = gray.mean()

    if brightness < 120:
        score -= 15
        explanations.append("Image appears darker than expected")
    else:
        explanations.append("Brightness within normal range")

    # ---------- 2. 色偏（是否偏黄） ----------
    r, g, b = img_np.mean(axis=(0, 1))
    yellow_bias = r - b  

    if yellow_bias > 20:
        score -= 20
        explanations.append("Noticeable yellow tint detected")
    else:
        explanations.append("No strong color shift detected")

    # ---------- 3. 清晰度（模糊/絮状） ----------
    laplacian_var = cv2.Laplacian(gray, cv2.CV_64F).var()

    if laplacian_var < 80:
        score -= 15
        explanations.append("Low image clarity (possible turbidity)")
    else:
        explanations.append("Image clarity within expected range")

    # ---------- 兜底 ----------
    score = max(40, min(score, 100))

    return score, explanations

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--csv", required=True)
    ap.add_argument("--opened_days", type=float, required=True)
    ap.add_argument("--stored_days", type=float, default=None)
    ap.add_argument("--img", default=None)
    args = ap.parse_args()

    time_score, time_reasons = compute_time_factor(args.opened_days, args.stored_days)
    env_score, env_reasons = compute_env_factor(args.csv)
    vis_score, vis_reasons = compute_visual_factor(args.img)

    # 加权（你图里的：50/30/20）
    final = (time_score * 0.5) + (env_score * 0.3) + (vis_score * 0.2)
    final = int(round(clamp(final)))

    # 给一个“灯号”方便前端显示
    if final >= 75:
        band = "green"
        label = "Safe"
    elif final >= 50:
        band = "yellow"
        label = "Watch"
    else:
        band = "red"
        label = "Risk"

    out = {
        "freshness_score": final,
        "band": band,
        "label": label,
        "breakdown": {
            "time": int(round(time_score)),
            "environment": int(round(env_score)),
            "visual": int(round(vis_score)),
            "weights": {"time": 0.5, "environment": 0.3, "visual": 0.2}
        },
        "top_reasons": (time_reasons[:1] + env_reasons[:2] + vis_reasons[:1])[:4]
    }

    print(json.dumps(out, ensure_ascii=False))

if __name__ == "__main__":
    main()
